import 'package:dio/dio.dart';
import 'package:razinsoft_rider/core/config/api_endpoints.dart';
import 'package:razinsoft_rider/data/services/api/dio_client.dart';

import '../../domain/interfaces/auth_service_interface.dart';
import 'local_storage_service.dart';

class AuthService implements IAuthService {
  final DioClient dioClient;
  AuthService({required this.dioClient});
  @override
  Future<Response> login(
      {required String phone,
      required String countryCode,
      String? deviceToken}) async => dioClient.dio.post(
      ApiEndpoints.loginUrl,
      data: {
        'mobile': phone,
        'country_code': countryCode,
        'device_token': deviceToken
      },
    );

  @override
  Future<Response> resendSignIn({required num? userId, required String? deviceToken}) async => await dioClient.dio.post(
      '${ApiEndpoints.resendSignIn}/$userId',
      data: {
        'device_token': deviceToken,
      }
  );

  @override
  Future<Response> loginWithPassword(
      {required String mobile,
      required String password,
      String? deviceToken, String? wantLogin}) async {
    final url = wantLogin == null ? ApiEndpoints.loginUrl : '${ApiEndpoints.loginUrl}?is_login=yes';
    return await dioClient.dio.post(
      url,
      data: {
        'mobile': mobile,
        'country_code': await LocalStorageService().getPhoneCode(),
        'password': password,
        'device_token': deviceToken
      },
    );
  }

  @override
  Future<Response> verifyOtp(
      {required String mobile,
      required String otp,
      String? deviceToken,
        String? wantLogin}) async {
    final url = wantLogin == null ? ApiEndpoints.loginUrl : '${ApiEndpoints.loginUrl}?is_login=yes';
    return await dioClient.dio.post(
      url,

      data: {
        'mobile': mobile,
        'country_code': await LocalStorageService().getPhoneCode(),
        'otp': otp,
        'device_token': deviceToken
      },
    );
  }

  @override
  Future<Response> updatePassword({required String password}) async => await dioClient.dio.post(ApiEndpoints.updatePassword, data: {
      'password': password,
      'password_confirmation': password,
    });

  @override
  Future<Response> changePassword(
      {required String currentPassword,
      required String newPassword,
      required newConfirmPassword}) async => await dioClient.dio.post(
      ApiEndpoints.changePassword,
      data: {
        'current_password': currentPassword,
        'password': newPassword,
        'password_confirmation': newConfirmPassword
      },
    );

  @override
  Future<Response> updateProfile({required Map<String, dynamic> data}) async => await dioClient.dio.post(
      ApiEndpoints.updateProfile,
      data: data,
    );

  @override
  Future<Response> resendOTP({required String mobile}) async => await dioClient.dio.post(ApiEndpoints.resendOTP, data: {
      'mobile': mobile,
    });

  @override
  Future<Response> forgotPassword({required String mobile}) async => await dioClient.dio.post(ApiEndpoints.forgotPassword, data: {
      'mobile': mobile,
    });

  @override
  Future<Response> updateProfilePhoto({required String imagePath}) async {
    final FormData formData = FormData.fromMap(
        {'profile_picture': await MultipartFile.fromFile(imagePath)});
    return await dioClient.dio.post(ApiEndpoints.updateProfilePhoto, data: formData);
  }

  @override
  Future<Response> riderDetails() async => await dioClient.dio.get(ApiEndpoints.riderDetails);

  @override
  Future<Response> logout() async => await dioClient.dio.get(ApiEndpoints.logout);

  @override
  Future<Response> forgetVerifyOtp(
      {required String mobile, required String otp}) async => await dioClient.dio.post(ApiEndpoints.forgetVerifyOtp, data: {
      'mobile': mobile,
      'otp': otp,
    });

  @override
  Future<Response> requestOTP({required String mobile}) async => await dioClient.dio.post(ApiEndpoints.requestOTP, data: {
      'mobile': mobile,
    });

  @override
  Future<Response> resetPassword({required Map<String, dynamic> data}) async => await dioClient.dio.post(ApiEndpoints.resetPassword, data: data);
}
